/**
 * \file: exchnd_interface.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * User space interface to the exception handler device.
 * The interface encapsulates the ioctls to the exception handler device into
 * functions to be used by the daemon.
 *
 * \component: exchndd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/ptrace.h>
#include <sys/types.h>
#include <sys/syscall.h>
#include <sys/wait.h>
#include <unistd.h>

#include <linux/exchnd.h>
#include "exchnd_interface.h"
#include "exchnd_collector.h"

/*
 * \func exchnd_get_version
 *
 * Retrieves the version of the exception handler kernel module via ioctl.
 *
 * \param exh_if Exception handler interface containing the fd for the ioctl
 *
 * \return Version of the exception handler kernel module
 */
/*PRQA: Lint Message 429: No custodial issue here.*/
/*lint -save -e429*/
unsigned long exchnd_get_version(ExchndInterface_t *exh_if)
{
    if (exh_if) {
        unsigned long value;
        int rc = ioctl(exh_if->efd,
                       IOCTL_EXCHND_VERSION, &value);
        return rc >= 0 ? value : (unsigned long)~0;
    }

    return 0;
}
/*lint -restore*/

/*
 * \func exchnd_create
 *
 * Creates an instance of the exception handler interface. The instance is
 * allocated inside the function and has to be deleted later.
 *
 * \param dev_name Name of the exception handler device
 *
 * \return Pointer to exception handler interface
 */
ExchndInterface_t *exchnd_create(const char *dev_name)
{
    ExchndInterface_t *exh_if =
        (ExchndInterface_t *)malloc(sizeof(ExchndInterface_t));

    if (!exh_if)
        return NULL;

    exh_if->PTRACE_attached = EXH_DETACHED;
    exh_if->efd = open(dev_name, O_RDONLY);

    if (exh_if->efd >= 0) {
        /* check the version of exchnd Linux kernel driver */
        unsigned long version = exchnd_get_version(exh_if);

        if ((version < EXCHND_REQUIRED_VERSION_MIN) ||
            (version > EXCHND_REQUIRED_VERSION_MAX)) {
            /* the driver version is insufficient */
            exchnd_print_error("%s: wrong version v%d.%02x (0x%lX), "
                               "expecting v%d.%02x ... v%d.%02x\n",
                               dev_name, version >> 8, version & 0xff, version,
                               EXCHND_REQUIRED_VERSION_MIN >> 8,
                               EXCHND_REQUIRED_VERSION_MIN & 0xff,
                               EXCHND_REQUIRED_VERSION_MAX >> 8,
                               EXCHND_REQUIRED_VERSION_MAX & 0xff);
            close(exh_if->efd);
            free(exh_if);
            exh_if = NULL;
        } else {
            exh_if->name = strdup(dev_name);
        }
    } else {
        /* failed to open the device */
        exchnd_print_error("Could not open exception handler device. Error %d",
                           exh_if->efd);
        free(exh_if);
        exh_if = NULL;
    }

    return exh_if;
}

/* exchnd_info - information about the exception handler */
void exchnd_info(ExchndInterface_t *exh_if, int fd)
{

    if (exh_if) {
        unsigned long version = exchnd_get_version(exh_if);

        dprintf(fd, "\n%s", exh_if->name);

        if (~version)
            dprintf(fd, " v%lu.%02lx", version >> 8, version & 0xff);

        dprintf(fd, "\n");
    }
}

/**
 * \func exchnd_process
 *
 * Reads one message and transports it to backends
 * Reads the message from the file delivered in the exception handle. It
 * writes the data to the backends and if necessary it delegates to the
 * data collection to the responsible functions which do additional
 * collections that are written to the backends.
 *
 * \param handle Handle of the used exception handler device
 * \param rd File descriptor set containing the file handles of the device
 * \param backend Linked list of backends
 */
int exchnd_process(ExchndInterface_t *exh_if)
{
    char *data = NULL;
    struct exchnd_message_header header;
    int len = 0;
    /* Insert command line in header once per exception. */
    static long prev_pid;
    int ret = EXIT_FAILURE;

    if (exh_if == NULL) {
        exchnd_print_error("No exchnd interface available !");
        return ret;
    }

#ifdef IOCTL_EXCHND_RECOVERY

    /* Ensure we are aligned */
    if (ioctl(exh_if->efd, IOCTL_EXCHND_RECOVERY) < 0) {
        exchnd_print_error("Unable to call recovery: %s", strerror(errno));

        if (errno == EACCES) {
            exchnd_print_error("No right to read on device."
                               " Please check if someone else is not"
                               " already there.");
            action |= EXCHND_TERMINATE;
            return ret;
        }
    }

#endif

    /* Read header */
    len = read(exh_if->efd, &header, sizeof(header));

    if (len != sizeof(header)) {
        /* Reading in blocking mode -> This must be an error case */
        exchnd_print_error("Reading incomplete header. Size/Error = %d", len);

        if (errno == EACCES) {
            exchnd_print_error("No right to read on device."
                               " Please check if someone else is not"
                               " already there.");
            exit(-1);
        }

        return ret;
    }

    exchnd_touch_wd(header.pid, header.type);

    if (header.flags.collected == 1) {
        data = malloc(header.length + 1);

        if (!data) {
            exchnd_print_error("%s: No memory available for data: %s (%d).",
                               __func__,
                               strerror(errno),
                               errno);
            return ret;
        }

        memset(data, 0, header.length + 1);

        len = read(exh_if->efd, data, header.length);

        if (len != (int)header.length) {
            /* Reading in blocking mode -> This must be an error case */
            exchnd_print_error("Reading incomplete data."
                               "Expected %d Size/Error = %d",
                               header.length,
                               len);
            free(data);
            return ret;
        }

        if (header.flags.internal == 0) {
            /* No internal message */
            exchnd_backend_store(data, header.length);

            if ((prev_pid != header.pid)
                && (header.type == EHM_NONE)
                && (header.pid != 0)) {
                /* EHM_NONE means we are printing the header. */
                prev_pid = header.pid;
                exchnd_add_commandline(&header);
            }
        }
    }

    if (header.flags.addition_needed == 1)
        /* Check collector is configured */
        if (collector_list[header.type] != NULL)
            collector_list[header.type](exh_if, &header, data);

    if (header.type != EHM_NONE)
        /* Now reset prev_pid as we can have an other
         * exception/event with the same pid.
         */
        prev_pid = 0;

    if ((header.type == EHM_ACTION_START) || (header.type == EHM_SYS_RESTART))
        ret = header.type;
    else
        ret = EXIT_SUCCESS;

    free(data);
    return ret;
}

/* exchnd_destroy - de-initialize the error memory interface */
void exchnd_destroy(ExchndInterface_t *exh_if)
{
    if (exh_if) {
        if (exh_if->efd > 0)
            close(exh_if->efd);

        if (exh_if->name)
            free(exh_if->name);

        free(exh_if);
    }
}
